import time
import app
import requests
from urllib3.exceptions import InsecureRequestWarning

requests.packages.urllib3.disable_warnings(category=InsecureRequestWarning)
import re
from urllib.parse import quote
import suds


def set_api_key_and_domain(appui_obj):
    current_key = appui_obj.settingsValue("chemairs_api_key") or ""
    current_domain = appui_obj.settingsValue("chemairs_api_domain")

    new_domain = appui_obj.query("API Domain",
                                 "Please input API domain (e.g. 'https://chemairs.chemical.ai'):", current_domain)

    if new_domain and not new_domain.startswith(('http://', 'https://')):
        new_domain = 'https://' + new_domain

    new_key = appui_obj.query("API Key", "Please input ChemAIRS API Key:", current_key)
    if not new_key:
        return

    appui_obj.updateSettingsValue("chemairs_api_key", new_key)
    appui_obj.updateSettingsValue("chemairs_api_domain", new_domain)
    appui_obj.show("API Configuration Saved Successfully")


def retro_synthesis_search(appui_obj):
    api_key = appui_obj.settingsValue("chemairs_api_key")
    api_domain = appui_obj.settingsValue("chemairs_api_domain")

    if not api_domain:
        return appui_obj.showError("Please set the ChemAIRS API Domain first")

    if not api_key:
        return appui_obj.showError("Please set the ChemAIRS API Key first")

    smiles = appui_obj.getCurrentMolecule()
    if not smiles:
        return appui_obj.showError("Please select a molecule")

    url = f"{api_domain}/open/v2/retro-task-start?smiles={quote(smiles)}"
    headers = {"X-Api-Key": api_key}

    try:
        response = requests.get(url, headers=headers, verify="py3/chemAIRS/chemical_ai_cert_chain.pem")
        response.raise_for_status()
        result = response.json()

        current_ds_bytes = appui_obj.importDS("")
        ds = app.Dataset()
        ds.parse(current_ds_bytes)

        url_col_index = ds.columnIndex("Retrosynthesis Link", 1)

        if url_col_index == -1:
            url_header = app.Dataset.meta("Retrosynthesis Link", app.TYPE_STRING, ())
            ds.insertColumn(url_header, [ds.string("")] * ds.rowSz(), ds.columnSz())
            url_col_index = ds.columnSz() - 1

        target_row = -1
        smiles_col_index = 0

        for col in range(ds.columnSz()):
            if ds.columnType(col) == app.TYPE_MOLECULE:
                smiles_col_index = col
                break

        for row in range(ds.rowSz()):
            current_smiles = ds.stringval(smiles_col_index, row)
            if current_smiles == smiles:
                target_row = row
                break

        if target_row == -1:
            appui_obj.showError("No matching molecular record found!")
            return

        url = f"<a href=\"{result['url']}\">ChemAIRS Link</a>"
        url_entry = app.Dataset.string(url)

        ds.set(url_col_index, target_row, url_entry)

        url_col_bytes = ds.packColumn(url_col_index)

        appui_obj.removeColumnFromDataSet("", url_col_index)
        success = appui_obj.addColumnToDataSet("", url_col_index, url_col_bytes)

        if success:
            appui_obj.show("The task has been submitted. Please access the results through the provided link.")

    except Exception as e:
        appui_obj.showError(f"Request failed: {str(e)}")


def sa_score_calculation(appui_obj):
    api_key = appui_obj.settingsValue("chemairs_api_key")
    api_domain = appui_obj.settingsValue("chemairs_api_domain")

    if not api_domain:
        return appui_obj.showError("Please set the ChemAIRS API Domain first")
    if not api_key:
        return appui_obj.showError("Please set the ChemAIRS API Key first")

    smiles_list = appui_obj.getCurrentMolecules()
    if not smiles_list:
        return appui_obj.showError("Please select a molecule")

    data = {
        "useELNData": True,
        "mode": 1,
        "materialSource": 0,
        "smiles": smiles_list,
        "callbackUrl": None
    }

    headers = {
        "X-Api-Key": api_key,
        "Content-Type": "application/json"
    }

    try:
        response = requests.post(
            f"{api_domain}/open/v2/sascore-json-search",
            json=data,
            headers=headers,
            verify="py3/chemAIRS/chemical_ai_cert_chain.pem"
        )
        response.raise_for_status()

        task_id = response.text.strip('"')

        current_ds_bytes = appui_obj.importDS("")
        ds = app.Dataset()
        ds.parse(current_ds_bytes)

        url_col_index = ds.columnIndex("SAScore Link", 1)
        if url_col_index == -1:
            url_header = app.Dataset.meta("SAScore Link", app.TYPE_STRING, ())
            ds.insertColumn(url_header, [ds.string("")] * ds.rowSz(), ds.columnSz())
            url_col_index = ds.columnSz() - 1

        smiles_col_index = -1
        for col in range(ds.columnSz()):
            if ds.columnType(col) == app.TYPE_MOLECULE:
                smiles_col_index = col
                break
        if smiles_col_index == -1:
            return appui_obj.showError("No molecule column found")

        url_entries = []
        link = f"<a href=\"{api_domain}/sascore/{task_id}\">ChemAIRS Link</a>"
        for row in range(ds.rowSz()):
            current_smiles = ds.stringval(smiles_col_index, row)
            if current_smiles in smiles_list:
                url_entries.append(app.Dataset.string(link))
            else:
                old_url = ds.stringval(url_col_index, row)
                url_entries.append(ds.string(old_url))

        ds.setColumn(url_col_index, url_entries)
        url_col_bytes = ds.packColumn(url_col_index)
        appui_obj.removeColumnFromDataSet("", url_col_index)
        success_sascore = appui_obj.addColumnToDataSet("", url_col_index, url_col_bytes)

        if success_sascore:
            appui_obj.show(
                "The task has been submitted.Please click on `Refresh SAScore Results` to obtain progress and results"
            )
            # while True:
            #     process = get_sa_score_results(appui_obj)
            #     parts = process.replace(" ", "").split("/")

            #     if len(parts) == 2 and parts[0] == parts[1]:
            #        appui_obj.show("SAScore results have been updated.")
            #        break;
            #     else:
            #        continue;
        else:
            appui_obj.showError("Failed to update dataset columns.")

    except Exception as e:
        appui_obj.showError(f"Request failed: {str(e)}")


def get_sa_score_results(appui_obj):
    api_key = appui_obj.settingsValue("chemairs_api_key")
    api_domain = appui_obj.settingsValue("chemairs_api_domain")
    if not api_domain:
        return appui_obj.showError("Please set the ChemAIRS API Domain first")
    if not api_key:
        return appui_obj.showError("Please set the ChemAIRS API Key first")

    current_ds_bytes = appui_obj.importDS("")
    ds = app.Dataset()
    ds.parse(current_ds_bytes)

    task_id_col_index = ds.columnIndex("SAScore Link", 1)
    if task_id_col_index == -1:
        return appui_obj.showError("SAScore Link column not found in the dataset.")

    task_links = set()
    for row in range(ds.rowSz()):
        task_link = ds.stringval(task_id_col_index, row).strip()
        if task_link:
            task_links.add(task_link)

    if not task_links:
        return appui_obj.showError("No Task Ids found in the dataset.")
    else:
        link = task_links.pop()

    match = re.search(r'href="[^"]*/([0-9a-fA-F-]{36})"', link)
    if match:
        task_id = match.group(1)
    else:
        return appui_obj.showError("Task Id error.")

    url = f"{api_domain}/open/v2/sascore-results/{task_id}/1000000/1"
    headers = {"X-Api-Key": api_key}
    progress = ""
    try:
        response = requests.get(url, headers=headers, verify="py3/chemAIRS/chemical_ai_cert_chain.pem")
        response.raise_for_status()
        results = response.json()
        progress = results.get("progress")
        result_map = {}
        for item in results.get("searchResponseDtos", []):
            smiles = item.get("smiles")
            if smiles:
                sa_score = str(item.get("saScore", ""))
                steps = str(item.get("steps", ""))
                estimated_cost = str(item.get("estimatedCost", ""))
                result_map[smiles] = {
                    "SAScore": sa_score,
                    "Steps": steps,
                    "EstimatedCost": estimated_cost
                }

        mol_col_index = -1
        for col in range(ds.columnSz()):
            if ds.columnType(col) == app.TYPE_MOLECULE:
                mol_col_index = col
                break
        if mol_col_index == -1:
            return appui_obj.showError("No molecule column found")

        sascore_entries = []
        steps_entries = []
        cost_entries = []
        for row in range(ds.rowSz()):
            current_smiles = ds.stringval(mol_col_index, row)
            if current_smiles in result_map:
                entry = result_map[current_smiles]
                sascore_entries.append(app.Dataset.cont(float(entry["SAScore"]), 0.0))
                steps_entries.append(app.Dataset.string(entry["Steps"]))
                cost_entries.append(app.Dataset.string(entry["EstimatedCost"]))
            else:
                sascore_entries.append(app.Dataset.cont(0.0, 0.0))
                steps_entries.append(app.Dataset.string(""))
                cost_entries.append(app.Dataset.string(""))

        def update_or_add_column(inner_ds, inner_appui_obj, col_name, entries):
            col_index = inner_ds.columnIndex(col_name, 1)
            if col_index == -1:
                if col_name == "SAScore":
                    header = inner_ds.createNumberHeader(col_name)
                else:
                    header = app.Dataset.meta(col_name, app.TYPE_STRING, ())
                inner_ds.insertColumn(header, entries, inner_ds.columnSz())
                col_index = inner_ds.columnSz() - 1
            else:
                inner_ds.setColumn(col_index, entries)
            col_bytes = inner_ds.packColumn(col_index)
            inner_appui_obj.removeColumnFromDataSet("", col_index)
            success = inner_appui_obj.addColumnToDataSet("", col_index, col_bytes)
            return success

        update_or_add_column(ds, appui_obj, "SAScore", sascore_entries)
        time.sleep(0.5)
        update_or_add_column(ds, appui_obj, "Steps", steps_entries)
        time.sleep(0.5)
        update_or_add_column(ds, appui_obj, "EstimatedCost", cost_entries)

        parts = progress.replace(" ", "").split("/")
        appui_obj.show(f"Current progress: {parts[0]}/{parts[1]}")

    except requests.exceptions.HTTPError as e:
        appui_obj.showError(f"HTTP Error: {e.response.status_code} - {e.response.text}")
    except requests.exceptions.RequestException as e:
        appui_obj.showError(f"Network request failed: {str(e)}")
    except Exception as e:
        appui_obj.showError(f"Error processing results: {str(e)}")

    return progress


def get_stardrop_definitions():
    return [
        {
            'script_name': 'ChemAIRS/Set API Configuration',
            'callback': set_api_key_and_domain
        },
        {
            'script_name': 'ChemAIRS/Retrosynthetic Search',
            'callback': retro_synthesis_search
        },
        {
            'script_name': 'ChemAIRS/SAScore Calculation',
            'callback': sa_score_calculation
        },
        {
            'script_name': 'ChemAIRS/Refresh SAScore Results',
            'callback': get_sa_score_results
        }
    ]
