# utils.py - helper functions to do miscellaneous things
# Use of this code is subject to the terms of the StarDrop License Agreement and use, copying or redistribution of this
# file for use by other persons or organisations is prohibited
# Copyright(C) Optibrium Ltd 2022
# stardrop-support@optibrium.com
#

import os
import re
import sys
import types
import traceback
from typing import Any, Callable, Union
from importlib import util as iutil
from inspect import getmembers, isfunction

import app

from stardrop_helper_functions import logfile_handling as logfile


"""

This functions within this script are supposed to be composed of functions which do not belong in any other
script, but are nonetheless useful enough to be reutilized across various scripts. This custom script is
not meant to be called directly from the StarDrop Custom Scripts menu.

"""

"""
The following block of code below creates a global variable called 'app_column_type_dict' which is meant to
be used in the 'get_column_type_str' function.
"""

pattern = re.compile('TYPE_')

app_column_types = [pattern.split(item)[1].lower() for item in dir(app) if pattern.search(item) is not None]

app_column_types.remove('unkown')
app_column_types.remove('mixed')
app_column_types.remove('molecule')
app_column_types.append('mol')
app_column_types.remove('flex_x')
app_column_types.append('flexx')

app_column_type_dict = \
    dict([(getattr(app, app_column_type), 'app.' + app_column_type) for app_column_type in app_column_types])


def get_stardrop_definitions_from_folder(folder_path: str,
                                         function_name: str) -> list[dict[str, Union[str, Callable[[Any], None]]]]:
    """
    Specify a folder with Python scripts to be imported into StarDrop scripts, and then create StarDrop
    definitions to pass on to a collector script which will pass those definitions on to StarDrop. The
    logfile_handling script should not be used within this function since this function is likely to be
    called before a logfile is initialized.
    @param: folder_path: The path of the folder with Python scripts to be imported into StarDrop
    @param: function_name: The name of the function to use as the main callback for each imported
                           Python script
    @returns: A list of StarDrop definitions
    """

    try:

        if not os.path.exists(folder_path):
            raise IOError(folder_path + ' does not exist')
        elif not os.path.isdir(folder_path):
            raise IOError(folder_path + ' exists, but is not a folder')
        elif os.path.dirname(os.path.realpath(traceback.extract_stack()[-2].filename)) == folder_path:
            print('\n\nIt is not generally a good idea to import scripts from the same folder as the collector script.\n\n')  # noqa: E501
        listings = os.listdir(folder_path)
        listings = [listing for listing in listings if os.path.splitext(listing)[1] == '.py']

        imported_modules = []
        for listing in listings:
            try:
                imported_modules.append(_load_pyfile_as_module(os.path.join(folder_path, listing)))
            except Exception:
                print('\nError loading ' + listing + ': ' + traceback.format_exc() + '\n')

        custom_modules = [module for module in imported_modules if function_name in [member[0] for member in getmembers(module, isfunction)]]  # noqa: E501

        definitions = \
            [{'script_name': os.path.basename(folder_path) + '/' + ' '.join([fragment.title() if not fragment.isupper() else fragment for fragment in module.__name__.split('_')]),  # noqa: E501
              'callback': getattr(module, function_name)} for module in custom_modules]

    except Exception as error:

        print('\n' + str(error) + '\n')
        definitions = []

    return definitions


def get_column_type_str(column_type: int) -> str:
    """
    Return a string which is the variable name of the dataset column type from the "app" module corresponding
    to the given StarDrop-defined integer. An exception will be thrown if an unknown column typeis given to
    this function.
    @param: column_type: the type of a column from the "app" module
    @returns: column_type_str: the corresponding column type as a string
    """

    if column_type not in app_column_type_dict:
        logfile.record_and_raise_exception(str(column_type) + ' is not a known StarDrop data type.')

    column_type_str = app_column_type_dict[column_type]

    return column_type_str


def get_qualifier_str(qualifier_number: int) -> str:
    """
    Parse the qualifer for a numerical entry in a StarDrop dataset into a string as defined in the
    app.Qualifiers class. Perhaps in the future, this function can be moved into the app.py script?
    @param: qualifier_number: The qualifer stored in the StarDrop dataset entry as a number
    @returns: qualifier_str: The corresponding string for the qualifier number
    """

    if   qualifier_number == app.Qualifiers.Unqualified:  # noqa: E271
        qualifier_str = ''
    elif qualifier_number == app.Qualifiers.LT:
        qualifier_str = '<'
    elif qualifier_number == app.Qualifiers.GT:
        qualifier_str = '>'
    elif qualifier_number == app.Qualifiers.PLUS:
        qualifier_str = '+'
    elif qualifier_number == app.Qualifiers.LTE:
        qualifier_str = '<='
    elif qualifier_number == app.Qualifiers.GTE:
        qualifier_str = '>='
    elif qualifier_number == app.Qualifiers.APPROX:
        qualifier_str = '~'

    return qualifier_str


def _load_pyfile_as_module(pyfile_path: str) -> types.ModuleType:
    """
    Programatically load a python script as module into Python. The logfile_handling script should not be
    used within this function since this function is likely to be called before a logfile is initialized.
    @param: pyfile_path: The path of the python script to be loaded as a module
    @returns: The corresponding module
    """

    pyfile_dir = os.path.dirname(pyfile_path)
    pyfile_name = os.path.splitext(os.path.basename(pyfile_path))[0]

    if not os.path.exists(pyfile_dir):
        raise IOError(pyfile_dir + ' does not exist.')
    else:
        if not os.path.isdir(pyfile_dir):
            raise IOError(pyfile_dir + ' exists, but is not a folder.')
        else:
            if not os.path.exists(pyfile_path):
                raise IOError(pyfile_path + ' does not exist, but ' + pyfile_dir + ' does exist.')
            else:
                if not os.path.isfile(pyfile_path):
                    raise IOError(pyfile_path + ' exists, but is not a file.')
                else:
                    if not os.path.splitext(pyfile_path)[1] == '.py':
                        raise IOError(pyfile_path + ' exists, but is not a python script.')

    spec = iutil.spec_from_file_location(pyfile_name, pyfile_path)
    module = iutil.module_from_spec(spec)
    sys.modules[pyfile_name] = module
    spec.loader.exec_module(module)

    return module
