# qt_widgets.py - functions to create and access PythonWidgets widgets
# Use of this code is subject to the terms of the StarDrop License Agreement and use, copying or redistribution of this
# file for use by other persons or organisations is prohibited
# Copyright(C) Optibrium Ltd 2022
# stardrop-support@optibrium.com
#

import PythonWidgets
from typing import Optional, Union

from stardrop_helper_functions import logfile_handling as logfile


"""
The classes within this script are supposed to allow any calling code to use available PythonWidgets
widgets with greater ease. This script is not meant to be called directly from the StarDrop
Custom Scripts menu.

This script is far from finished, and I definitely have plans to change this script and/or
expand upon it in the future. However, what is in here right now will serve as an MVP
(Minimum Viable Product) for the Helper Functions API v1.0
"""


class PromptGUIs:
    """
    The functions within this script create widgets which are meant to query the user, and/or
    make the user choose from amongst a set of options. One particular quirk all of these
    widgets have in common (aside from querying the user) is that if the user cancels out of
    selecting any options, then the function returns "None" or "False" as a value.
    """

    @staticmethod
    def select_column(column_names: list[str],
                      title: str,
                      label: str = None) -> Optional[str]:
        """
        Prompt the user to select a column from amongst a set of columns. Will return "None" if
        the user cancels out of selecting any columns.
        @param: column_names: The set of column names the user needs to choose from
        @param: title: The title of the column selection window
        @param: label: A message describing the purpose of selecting a column
        @returns: A boolean stating whether or not the user clicked cancel, as well
                  as the name of the column they picked (if they didn't click cancel)
        """

        column_select_window = PythonWidgets.ColumnPicker()
        column_select_window.setColumns(dict(zip(column_names, [0]*len(column_names))))
        column_select_window.setWindowTitle(title)

        if label is not None:
            column_select_window.setLabel(label)

        user_selected_a_column_flag = column_select_window.run()
        column_name = column_select_window.getColumn() if user_selected_a_column_flag else None

        user_selected_a_column_flag = column_name != '' and user_selected_a_column_flag
        selected_column_name = column_name if user_selected_a_column_flag else None

        return selected_column_name

    @staticmethod
    def select_option(options: list[str],
                      title: str,
                      label: str,
                      only_one_option_flag: bool = True,
                      default_options: Union[str, list[str]] = None) -> tuple[bool, Optional[Union[str, list[str]]]]:
        """
        Prompt the user to select a radio button option from amongst a set of radio button
        options. Will return "None" if the user cancels out of selecting any options.
        @param: column_names: The set of options the user needs to choose from
        @param: title: The title of the option selection window
        @param: label: A message describing the purpose of selecting an option
        @param: only_one_option_flag: A boolean stating whether or not the user can
                                      choose multiple options
        @param: default_options: The default option which appears to be chosen when the
                                 window pops up before the user has the opportunity to
                                 choose something
        @returns: A boolean stating whether or not the user clicked cancel, as well
                  as the name of the option(s) they picked (if they didn't click cancel)
        """

        option_select_window = PythonWidgets.OptionSelector(only_one_option_flag)
        option_select_window.setWindowTitle(title)
        option_select_window.setLabel(label + ': ')
        option_select_window.setAvailableItems(options)

        if default_options:

            if isinstance(default_options, str):
                default_options = [default_options]

            if not (len(default_options) > 1 and only_one_option_flag):
                if all([default_option in options for default_option in default_options]):
                    option_select_window.setSelectedItems(default_options)
                else:
                    logfile.record_warning('Not all default options exist in regular options, did not set any default options')  # noqa: E501
            else:
                logfile.record_warning('Gave multiple default options for select_option() when only one item can be selected at a time, did not set any default options')  # noqa: E501

        user_selected_any_option = option_select_window.run()
        option_selector_func = option_select_window.getSelectedItem if only_one_option_flag else option_select_window.getSelectedItems  # noqa: E501
        selected_options = option_selector_func() if user_selected_any_option else None

        return selected_options

    @staticmethod
    def select_parameters(title: str,
                          text: str,
                          possible_parameters: dict[str, list[str]]) -> Optional[dict[str, str]]:
        """
        Prompt the user to select values for a parameter (or multiple parameters). Will return "None"
        if the user cancels out of selecting any parameters.
        @param: title: The title of the parameter selection window
        @param: text: The message describing the overall purpose of choosing each parameter
        @param: possible_parameters: A mapping of parameter names to possible parameter values
        @returns: A boolean stating whether or not the user clicked cancel, as well
                  as the name of the parameter(s) they picked (if they didn't click cancel)
        """

        cb = PythonWidgets.ComboBoxSelector()
        cb.setWindowTitle(title)
        cb.setText(text)
        for idx, (parameter_name, possible_parameter_values) in enumerate(possible_parameters.items()):
            if idx == 0:
                cb.setOptions(parameter_name + ':', possible_parameter_values)
            else:
                cb.addOptions(parameter_name + ':', possible_parameter_values)

        user_selected_parameters_flag = cb.run()
        selected_parameters = dict([(parameter_name, cb.getSelectedOption(idx)) for idx, parameter_name in enumerate(possible_parameters)]) if user_selected_parameters_flag else None  # noqa: E501

        return selected_parameters

    @staticmethod
    def select_items_from_list(title: str,
                               explanation: str,
                               available_items: list[str],
                               multiple_selection_mode: bool) -> Optional[Union[str, list[str]]]:
        """
        Prompt the user to select an item (or items) from a list of items. Will return "None" if
        the user cancels out of selecting any items.
        @param: title: The title of the item-selection window
        @param: explanation: A message describing the purpose of selecting an item (or items)
        @param: available_items: The set of available items to choose from
        @param: multiple_selection_mode: Whether or not the user can choose multiple items
        @returns: A boolean stating whether or not the user clicked cancel, as well
                  as the name of the item(s) they picked (if they didn't click cancel)
        """

        sl = PythonWidgets.SimpleList()
        sl.setWindowTitle(title)
        sl.setLabel(explanation)
        sl.setAvailableItems(available_items)

        if multiple_selection_mode:
            sl.setMultipleSelections()

        user_selected_an_item_flag = sl.run()

        if user_selected_an_item_flag:
            selected_items = list(sl.getSelectedItems())

            if not multiple_selection_mode:
                selected_items = selected_items[0]

        else:
            selected_items = None

        return selected_items

    @staticmethod
    def select_continuation_status(message: str, title: str) -> bool:
        """
        Prompt the user as to whether or not they would like to continue executing the script or if they would
        like to cancel out of the current script instead.
        @param: message: The warning text which the user needs to read
        @param: title: The title of the warning window
        @returns: Whether or not the user would like to continue executing the script
        """

        warning_window = PythonWidgets.Warning()
        warning_window.setWindowTitle(title)
        warning_window.setMessage(message)
        continue_bool = warning_window.run()

        return continue_bool


class DisplayGUIs:
    """
    The functions within this class create widgets which are only meant to display some information
    to the user. Unlike with the PromptGUI class, these widgets don't return any outputs whatsoever
    (since they don't have to).
    """

    @staticmethod
    def warning_window(message: str, title: str) -> None:
        """
        Display a warning to the user.
        @param: message: The warning text which the user needs to read
        @param: title: The title of the warning window
        """

        warning_window = PythonWidgets.Warning()
        warning_window.setWindowTitle(title)
        warning_window.setMessage(message)
        _ = warning_window.run()

    @staticmethod
    def text_window(text_to_display: str, label: str, title: str) -> None:
        """
        Display some text to the user, which they can then copy to their clipboard if they need to.
        @param: text_to_display: The text which the user needs to read, and can also copy
                                 to their clipboard
        @param: label: A label which describes the content of the text (should probably be brief)
        @param: title: The title of the text display window
        """

        text_window = PythonWidgets.TextDisplay()
        text_window.setWindowTitle(title)
        text_window.setLabel(label + ':')
        text_window.setText(text_to_display)
        text_window.exec()


class SpecializedGUIs:
    """
    The functions within this class create and provided access to widgets which do not quite belong in
    any other class in this script.
    """

    @staticmethod
    def initialize_progress_window(title: str,
                                   message: str,
                                   max_progress_iterator: int) -> tuple[int, PythonWidgets.SimpleProgress]:
        """
        Return an initialized widget which the calling code can use to display a progress bar which
        tells the user how much progress has been made on a discrete and measurable series of tasks.
        The calling code will have to manually update the window as the tasks progress.
        @param: title: The title of the progress window
        @param: message: The message used to describe what kind of progress is taking place
        @param max_progress_iterator: The maximum number of measurable tasks which the progress
                                      should be binned into
        @return: An initialized progress iterator which starts at zero, and the actual progress
                 window object itself
        """

        progress_iterator = 0

        progDlg = PythonWidgets.SimpleProgress()
        progDlg.modal = True
        progDlg.setWindowTitle(title)
        progDlg.setMessage(message)
        progDlg.setMaximum(max_progress_iterator)
        progDlg.setProgress(progress_iterator)

        return [progress_iterator, progDlg]
