# LogfileHandling.py - helper functions to create and write to a logfile
# Use of this code is subject to the terms of the StarDrop License Agreement and use, copying or redistribution of this
# file for use by other persons or organisations is prohibited
# Copyright(C) Optibrium Ltd 2022
# stardrop-support@optibrium.com
#

import os
import logging
from typing import Union
from enum import Enum

from stardrop_helper_functions.qt_widgets import DisplayGUIs

"""

This functions within this script are supposed to compose an API which provides for a specialized implementation
of the logging module. This custom script is not meant to be called directly from the StarDrop Custom Scripts menu.

"""


stardrop_helper_functions_API_version = '1.2'
ACTIVE_HANDLERS = []


class LoggingLevel(Enum):
    DEBUG = logging.DEBUG
    INFO = logging.INFO
    WARNING = logging.WARNING
    ERROR = logging.ERROR
    CRITICAL = logging.CRITICAL


class LoggedRuntimeError(RuntimeError):
    """
    This exception should be raised for CRITICAL-level errors which have already been
    recorded in the log file.
    """
    pass


def record_and_raise_exception(err: Union[str, Exception]) -> None:
    """
    Record an error within the log file, display it to the user and raise an Exception. This function is only to meant
    to be used in situations where the calling script is not capable of using a try-catch statement to handle whatever
    error is generated, and so therefore an exception must be used to crash the program as well as to warn the user\
    that a crash occurred.
    @ param: err: The error message or exception to be displayed
    """

    err_is_exception = isinstance(err, Exception)
    err_is_logged_exception = isinstance(err, LoggedRuntimeError) if err_is_exception else False
    err_is_str = isinstance(err, str) if not err_is_exception else False

    if err_is_logged_exception:
        raise err

    else:

        err_msg = str(err) if err_is_exception else err

        DisplayGUIs.warning_window(err_msg, 'Error!')

        logging.critical(err_msg + '\n\nsee scripting console for more details' + '\n')

        if err_is_str:
            raise LoggedRuntimeError(err_msg)
        elif err_is_exception:
            raise err


def record_error(err_msg: str) -> None:
    """
    Record an error within the log file, and print it to the scripting console.
    @ param: err_msg: The error message to be displayed
    """

    print('\n' + err_msg + '\n')
    logging.error(err_msg)


def record_warning(warning: str) -> None:
    """
    Record a potential issue within the log file.
    @param: warning: The warning to be conveyed to the user
    """

    logging.warning(warning)


def record_info(info: str) -> None:
    """
    Record information for the user within the log file.
    @param: info: The information to be conveyed to the user
    """

    logging.info(info)


def record_debugging(debug_msg: str) -> None:
    """
    Record debugging messages in case more information is needed.
    @param: debug_msg: The debugging information to be conveyed to the user
    """

    logging.debug(debug_msg)


def initialize_logfile(script_name: str,
                       script_version: str,
                       log_folder_path: str,
                       log_file_name: str = 'app',
                       logging_level: LoggingLevel = None) -> None:
    """
    Initialize a log file with the specified name and folder. Will create a logfile if none exists,
    and will overwrite it if a new StarDrop session is started in between script function calls. Will
    append to logfile otherwise.
    @param: script_name: The name of the calling script for which the logfile is being initialized
    @param: script_version: The latest version of the calling script
    @param: log_folder_path: The folder/directory path of the log file to be initialized
    @param: log_file_name: The name of the log file to be initialized
    @param: logging_level: The logging level of the logfile to be initialized
    """

    if not logging_level:
        logging_level = LoggingLevel.INFO

    if os.path.exists(log_folder_path):

        # Construct the path of the relevant log file for the calling script
        logfile_path = os.path.join(log_folder_path, log_file_name + '.log')

        # Get the Root Logger
        root_logger = logging.getLogger()

        # Remove all handlers from the Root Logger
        for handler in root_logger.handlers:
            root_logger.removeHandler(handler)

        # Determine whether or not the logfile should be created/restarted
        create_or_restart_logfile = (not os.path.exists(logfile_path)) or (logfile_path not in [handler.baseFilename for handler in ACTIVE_HANDLERS])  # noqa: E501

        if create_or_restart_logfile:
            # If the logfile does need to be created/restarted...

            logfile_descriptions = \
                {'Main Script': f'{script_name}" version {script_version}',
                 'Dependency #1': f'"Helper Functions" API version: {stardrop_helper_functions_API_version}',
                 'Logging Level': logging_level.name}

            # Call the basicConfig() function which will create or overwrite the log file
            logging.basicConfig(filename=logfile_path,
                                filemode='w',
                                format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
                                datefmt='%Y-%m-%d %H:%M:%S',
                                level=logging_level.value)

            # Place description at beginning of logfile
            logging.info('Beginning logfile:\n\n\t\t\t' + '\n\t\t\t'.join([f'{name}: {value}' for name, value in logfile_descriptions.items()]) + '\n')  # noqa: E501

            # place the newly created File Handler for the calling script in the ACTIVE_HANDLERS list
            ACTIVE_HANDLERS.append(root_logger.handlers[0])

        else:
            # Otherwise, place the File Handler corresponding to the current script in the Root Logger (which should
            # have no other File Handlers since we got rid of them earlier)
            root_logger.addHandler([handler for handler in ACTIVE_HANDLERS if handler.baseFilename == logfile_path][0])

        logging.info('Beginning script')

    else:

        err_msg = f'The specified folder for the logfile does not exist: {log_folder_path}'
        DisplayGUIs.warning_window(err_msg + '\n\nNo logfile was created.\n',
                                   'Error!')
        raise RuntimeError(err_msg)


def finish_logfile() -> None:
    """
    Record that the script's function call is done within the logfile.
    """

    logging.info('Finishing script\n')
